% Determination location of the stagnation points of double well system in 
% confined aquifer with pumping and recharge well, , after daCosta 
% and Bennet (1960).
% Given:
% - Angle alpha of the flow direction with the x-axis,
% - Pumping/recharge rate Q,
% - Half distance a between the well located on the x-axis,
% - Aquifer thickness m,
% - Darcy velocity v0 in x-direction.
% Pumping and recharge well are located on the x-axis with coordinates 
% (-a, 0); (a, 0).
% Subsequent computation of stream function values at the stagnation points 
% and at the origin and calculation of the interflow rate, after daCosta 
% and Bennet (1960).
%
% Parameter input is performed by a dialogue window with default values. If
% the input value is not consistent (e.g., negative time), the dialogue 
% window is again active. Use consistent units consistently (e.g., use 
% always m, d). Output is performed to to the command window.
%
% Recirculation rate and fradction are calculated for both stagnation
% point. The results should be identical.
%
% 19 Nov. 2012                                 F. Stauffer IfU, ETH Zurich
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                          
% Input of parameters
test = 1;
while any(test)  % Test numerical input
    prompt = {'Pumping/recharge rate [m3/d] (>0):','Half distance between wells [m] (>0):','Aquifer thickness [m] (>0):','Darcy velocity in x-direction [m/d] (>0):','Angle of flow direction with x-axis []:'};
    dlg_title = 'Flow parameters';
    num_lines= 1;
    % Default values; Q, a, m, q0, alpha
    %    def     = {'3.14159265','1.0','1.0','1.0','0.0'};
    def     = {'1000','100.0','10.0','1.0','0.0'};

    stroemungsparameter  = inputdlg(prompt,dlg_title,num_lines,def);
    Q = str2num(stroemungsparameter{1}); QTrue = isempty(Q); QTrue2=(Q<=0);
    a = str2num(stroemungsparameter{2}); aTrue = isempty(a); aTrue2=(a<=0);
    m = str2num(stroemungsparameter{3}); mTrue = isempty(m); mTrue2=(m<=0);
    q0 = str2num(stroemungsparameter{4}); q0True = isempty(q0); q0True2=(q0<0);
    alpha = str2num(stroemungsparameter{5});
    test = [QTrue; aTrue; mTrue; q0True; QTrue2; aTrue2; mTrue2; q0True2];
end % while
str = [' ']; disp(str);
str = ['Parameter Q, a, m, q0, alpha:  ',num2str(Q),'  ',num2str(a),'  ',num2str(m),'  ',num2str(q0),'  ',num2str(alpha)]; disp(str);
alpha = 2*pi/360*alpha; % Transform angle from degree into rad
%
% Dimensionless pumping rate chi:
chi = Q/(pi*a*m*q0);
str = ['Dimensionless pumping rate:    ',num2str(chi)]; disp(str);
%
% Calculation of location of stagnation points S1(-xs,+ys) and S2(+xs/-ys)
arg3 = 1 + chi*chi - 2*chi*cos(alpha);
if arg3<0; arg3=0; end;
arg1 = 1 - chi*cos(alpha) + sqrt(arg3);
if arg1<0; arg1=0; end;
arg2 = chi*cos(alpha) - 1 + sqrt(arg3);
if arg2<0; arg2=0; end;
xs = a*sqrt(0.5*arg1);
ys = a*sqrt(0.5*arg2);
xs1 = -xs; 
ys1 = ys; 
str = ['Stagnation point S1 (xs1,ys1): ', num2str(xs1), '  ',num2str(ys1)]; disp(str);
xs2 = xs; 
ys2 = -ys;
str = ['Stagnation point S2 (xs2,ys2): ', num2str(xs2), '  ',num2str(ys2)]; disp(str);%
%
% Calculation of the stream function at stagnation point S1
thetar1 = atan2(ys1,(xs1-a));
thetap1 = atan2(ys1,(xs1+a));
% Stream function at stagnation point S1:
psi1 = Q/m*(-1/(pi*chi)*(ys1/a*cos(alpha)-xs1/a*sin(alpha))+1/(2*pi)*(thetap1-thetar1));
str = ['Stream function at S1:         ', num2str(psi1)]; disp(str);
% Stream function at origin (y=0+):
psi01 = -Q/(m*2);
str = ['Stream function at origin:     ', num2str(psi01)]; disp(str);
% Recirculation rate I1 and fraction f1:
i1 = 2*(psi1 - psi01); 
str = ['Recirculation rate 1:          ', num2str(i1)]; disp(str);
f1 = i1/Q;
str = ['Fraction of pumping rate 1:    ', num2str(f1)]; disp(str);
%
% Calculation of the stream function at stagnation point S2
thetar2 = atan2(ys2,(xs2-a));
thetap2 = atan2(ys2,(xs2+a));
% Stream function at stagnation point S2:
psi2 = Q/m*(-1/(pi*chi)*(ys2/a*cos(alpha)-xs2/a*sin(alpha))+1/(2*pi)*(thetap2-thetar2));
str = ['Stream function at S2:         ', num2str(psi2)]; disp(str);
% Stream function at origin (y=0-):
psi02 = +Q/(m*2);
str = ['Stream function at origin:     ', num2str(psi02)]; disp(str);
% Recirculation rate I2 and fraction f2:
i2 = 2*(psi02 - psi2);
str = ['Recirculation rate 2:          ', num2str(i2)]; disp(str);
f2 = i1/Q;
str = ['Fraction of pumping rate 2:    ', num2str(f2)]; disp(str);