% Program Wells_in_flow_field.m 
%
% developed with Matlab 6.5
%
% Analytical calculation of contour lines of head and stream function of 
% well systems in uniform unlimited flow field of a confined aquifers.
% Without wells the flow field has a positive flow direction with positive 
% flow gradient from west to east. 
%
% The analytical solution is obtained by superposition of a regional flow
% field and several wells for nx*ny grid points. nx=401
%
% Pumping well: Rate QW < 0; Recharge well: Rate QW > 0.
% The stream function shows a singularity at (Step by QW). This step is
% represented by a thick blue line. this thick line is not necessarily a
% stream line.
%
% The wells have a well radius rw>0. If a grid point coincides with a well
% the head is calculated at a distance x+rw.
%
% Parameter input is performed by a dialogue window with default values. If
% the input value is not consistent (e.g., negative time), the dialogue 
% window is again active. Use consistent units (e.g., use always m, d).
%
% Version 24. Nov. 2012,                     Fritz Stauffer, IfU ETH Zrich
%**************************************************************************
%
% Input: Limits for graphical representation
test = 1;
while any(test) % Test numerical input
   prompt = {'Minimum x-coordinate xmin:','Maximaum x-coordinate xmax>xmin:','Minimum y-coordinate ymin:','Maximaum y-coordinate ymax>ymin:'};
   dlg_title = 'Presented domain';
   num_lines= 1;
   def     = {'-100','100','-60','60'};
   geometrie  = inputdlg(prompt,dlg_title,num_lines,def);
   xmin = str2num(geometrie{1}); xminTrue = isempty(xmin);
   xmax = str2num(geometrie{2}); xmaxTrue = isempty(xmax); xmaxTrue2=(xmax<=xmin);
   ymin = str2num(geometrie{3}); yminTrue = isempty(ymin);
   ymax = str2num(geometrie{4}); ymaxTrue = isempty(ymax); ymaxTrue2=(ymax<=ymin);
   test = [xminTrue; xmaxTrue; yminTrue; ymaxTrue; xmaxTrue2; ymaxTrue2];
end % while
%
% Input: Flow parameters
test = 1;
while any(test)  % Test numerical input
    prompt = {'Hydraulic gradient (positiv for flow from west to east:','Hydraulic conductivity (>0):','Aquifer thickness (>0):','Number of wells: (>-1)'};
    dlg_title = 'Flow parameters';
    def     = {'0.005','20.0','10.0','2'};
    stroemungsparameter  = inputdlg(prompt,dlg_title,num_lines,def);
    gefaelle = str2num(stroemungsparameter{1}); gefaelleTrue = isempty(gefaelle);
    kf = str2num(stroemungsparameter{2}); kfTrue = isempty(kf); kfTrue2=(kf<=0);
    m = str2num(stroemungsparameter{3}); mTrue = isempty(m); mTrue2=(m<=0);
    nwells = str2num(stroemungsparameter{4}); nwellsTrue = isempty(nwells); nwellsTrue2=(nwells<0);
    test = [gefaelleTrue; kfTrue; mTrue; nwellsTrue; kfTrue2; mTrue2; nwellsTrue2];
end % while
%
% Input: Coordinates xw, yw, Recharge/pumping rate qw, Well radius rw, Radius of influence of well ri
xw = zeros(nwells,1); yw = zeros(nwells,1);
qw = zeros(nwells,1); ri = zeros(nwells,1);
for k=1:nwells;
    test = 1;
    while any(test) % Test numerical input
        prompt = {'x-coordinate of well:','y-coordinate of well:','Recharge/pumping rate (pumping: <0; recharge:>0):','Well radius (>0):','Radius of influence of well (>0):'};
        dlg_title = strcat('Parameters well_',num2str(k));
        if k==1; def = {'32.0','0.0','100.0','0.1','100.0'}; else;
        def     = {'-32.0','0.0','-100.0','0.1','100.0'}; end;
        brunnenparameter  = inputdlg(prompt,dlg_title,num_lines,def);
        xwk = str2num(brunnenparameter{1});  xwTrue = isempty(xwk);
        ywk = str2num(brunnenparameter{2}); ywTrue = isempty(ywk);
        qwk = str2num(brunnenparameter{3}); qwTrue = isempty(qwk);
        rwk = str2num(brunnenparameter{4}); rwTrue = isempty(rwk); rwTrue2=(rwk<=0);
        rik = str2num(brunnenparameter{5}); riTrue = isempty(rik); riTrue2=(rik<=0);
        test = [xwTrue; ywTrue; qwTrue; rwTrue; riTrue; rwTrue2; riTrue2];
    end % while
    xw(k) = xwk;
    yw(k) = ywk;
    qw(k) = qwk;
    rw(k) = rwk;
    ri(k) = rik;
end;
%
% Initialize head field, stream function field, x- and y-axes
nx = 401;
ny = ceil(nx/(xmax-xmin)*(ymax-ymin));
phi = zeros(ny,nx); psi = zeros(ny,nx);
xax=zeros(nx,1); yax=zeros(ny,1);
% Calculate head and stream function
xm = (xmin+xmax)*0.5;
ym = (ymin+ymax)*0.5;
xax = xmin:((xmax-xmin)/(nx-1)):xmax;
yax = ymin:((ymax-ymin)/(ny-1)):ymax;
yax=yax.';
x = repmat(xax,ny,1);
y = repmat(yax,1,nx);
phi = -gefaelle*(x-xm);
psi = kf*gefaelle*m*(y-ym);
q0 = kf*gefaelle*m;
for k=1:nwells
    r = sqrt((x-xw(k)).*(x-xw(k))+(y-yw(k)).*(y-yw(k)));
    r(find(r==0.0))=rw(k);
    phi = phi - qw(k)/(2.0*3.14159*kf*m)*log(r/ri(k));
    psi = psi + qw(k)/(2.0*3.14159)*atan2((y-yw(k)),(x-xw(k)));
end;
%
% Input for head and stream function contours
test = 1;
while any(test) % Test numerical input
    prompt = {'Minimaum head >-inf:','Maximum head <inf:','Step for head contours (>0):','Labels?','Minimum stream function:','Maximum stream function:','Stream function step (>0):','Labels?'};
    dlg_title = 'Head contours';
    hmin = min(min(phi(find(phi>-inf)))); 
    hmax = max(max(phi(find(phi<inf)))); 
    psimin1 = min(min(psi)); 
    psimax1 = max(max(psi)); 
    psimin = -abs(qw(1))*abs(floor(psimin1/abs(qw(1))));
    psimax = abs(qw(1))*ceil(psimax1/abs(qw(1)));
    def     = {num2str(hmin),num2str(hmax),num2str((hmax-hmin)/21),'no',num2str(psimin),num2str(psimax),num2str(abs(qw(1))/20),'no'};
    isolines  = inputdlg(prompt,dlg_title,num_lines,def);
    hmin = str2num(isolines{1}); hminTrue = isempty(hmin);
    hmax = str2num(isolines{2}); hmaxTrue = isempty(hmax); hmaxTrue2=(hmax<=hmin);
    dh = str2num(isolines{3}); dhTrue = isempty(dh); dhTrue2=(dh<=0);
    strlabelh = isolines{4};
    psimin = str2num(isolines{5}); psiminTrue = isempty(psimin);
    psimax = str2num(isolines{6}); psimaxTrue = isempty(psimax); psimaxTrue2=(psimax<=psimin);
    dpsi = str2num(isolines{7}); dpsiTrue = isempty(dpsi); dpsiTrue2=(dpsi<0);
    strlabelpsi = isolines{8};
    test = [hminTrue; hmaxTrue; dhTrue; psiminTrue; psimaxTrue; dpsiTrue; hmaxTrue2; dhTrue2; psimaxTrue2; dpsiTrue2];
end % while
%
% Show in Command Window
disp(strcat('Minimum head            =',num2str(hmin)));
disp(strcat('Maximum head            =',num2str(hmax)));
disp(strcat('Head step               =',num2str(dh)));
disp(strcat('Minimum stream function =',num2str(psimin)));
disp(strcat('Maximum stream function =',num2str(psimax)));
disp(strcat('Stream function step    =',num2str(dpsi)));
%
% Graphics of Head contours and streamlines
vh = hmin:dh:hmax;
vpsi = psimin:dpsi:psimax;
figure;
hold on;
    [xgraph,ygraph] = meshgrid(xax,yax);
    [C,h] = contour(xgraph,ygraph,phi,vh,'r');
    sc = strcmp(strlabelh,'yes');
    if sc==1; clabel(C,vh); end;
    [C,h] = contour(xgraph,ygraph,psi,vpsi,'b');
    sc = strcmp(strlabelpsi,'yes');
    if sc==1; clabel(C,vpsi); end;
    axis equal;
    axis([xmin xmax ymin ymax]);
    title('Potential lines (red) and stream lines (blue)');
    xlabel('x');
    ylabel('y');
hold off;
